-- World of Warcraft Addon
-- TargetNotes
-- author:  Tokipin@Mannoroth
--
-- glue for search window

TARGETNOTES_SETTINGS = TARGETNOTES_SETTINGS or {};
TARGETNOTES_SETTINGS["number of result rows"] = 9;
TARGETNOTES_SETTINGS["number of note rows"] = 10;

local num_results, num_notes;

local SearchWindow = CreateFrame( "frame", "TARGETNOTES_SEARCH", UIParent, "frame:search-window" );

local headings = {
    { text = "name", width = 112, margin = 22 },
    { text = "note", width = 187, margin = 3 },
    { text = "guild", width = 127, margin = 3 },
    { text = "faction", width = 66, margin = 3 },
    { text = "race", width = 63 },
    { text = "class", width = 64 },
    { text = "delete", width = 67, margin = 3 } };

local get_row_template = function( ... )
    local tbl = {};
    for i = 1, #headings do
        local val = select( i, ... );

        val = type(val)=="table" and val or { text = val };
        val["width"] = headings[i]["width"];
        val["margin"] = headings[i]["margin"];

        table.insert( tbl, val );
    end

    tbl[7]["backdrop"] = { bgFile = [[Interface\TutorialFrame\TutorialFrameBackground]] };
    tbl[7]["inherits"] = "button:window-button-template";

    return tbl;
end

local get_note_row_template = function( val, del )
    val = type(val)=="table" and val or { text = val };
    val["width"] = 628;
    val["margin"] = 22;

    del = type(del)=="table" and del or { text = "" };
    del["width"] = headings[7]["width"];
    del["margin"] = headings[7]["margin"];
    del["backdrop"] = { bgFile = [[Interface\TutorialFrame\TutorialFrameBackground]] };
    del["inherits"] = "button:window-button-template";

    return { val, del };
end

local blank_row = get_row_template();
local blank_note_row = get_note_row_template();
local tabulate_notes;

-- set up search window headings
for index, heading in ipairs( headings ) do
    heading["inherits"] = "button:window-button-template";
    heading["backdrop"] = { bgFile = [[Interface\TutorialFrame\TutorialFrameBackground]] };
end

SearchWindow:PrintRow( 0, headings );

local highlights = {};

local tabulate_result = function( index, result, DB, search_again )
    local e = setmetatable( result["entry"], { __call = function( tbl, key )
        return (key=="note") and tbl[2][#tbl[2]] or tbl[1][key];
    end } );

    local btn_handlers = (e"player or npc" == "player") and {
        ["onshow"] = function( self )
            self:RegisterForClicks( "LeftButtonUp", "RightButtonUp" );
        end,
        ["onclick"] = function( self, str_btn )
            if( str_btn == "RightButton" ) then
                local menu = SearchWindow.popup_menu;
                menu:Appear( self );
                menu.button:SetScript( "onclick", function( self )
                    SendWho('n-"' .. e"name" .. '"' );
                    menu:Disappear();
                end );
            end
        end
    };

    local row = get_row_template(
        { text = e"name", handlers = btn_handlers },
        { text = e"note", handlers = btn_handlers },
        { text = e"guild", handlers = btn_handlers },
        { text = e"faction", handlers = btn_handlers },
        { text = e"race", handlers = btn_handlers },
        { text = e"class", handlers = btn_handlers },

        --[[ entry delete button handlers ]]--
        { text = "",
          handlers = {
            ["onclick"] = function()
                DB:RemoveEntry( result["id"] );

                TARGETNOTES_FUNCTIONS["update_target_notes"]();
                SearchWindow:ClearRows( 1, math.huge );
                search_again();
            end,
            ["onenter"] = function( self ) self:SetText( "[delete]" ) end,
            ["onleave"] = function( self ) self:SetText( "" ) end } }
    );

    --[[ whole row handlers ]]--
    row.handlers = {
        ["onclick"] = function( self, str_btn )
            if( str_btn == "RightButton" ) then return end; -- return

            SearchWindow.popup_menu:Disappear()

            --[[ highlighting ]]--
            local count = 0;
            for btn in pairs( SearchWindow:GetRow( self.row ) ) do
                if not( btn.not_visible ) then
                    local strong_highlight = SearchWindow.getframe["sh"..count]["button:highlight"];
                    strong_highlight:SetFrameLevel( btn:GetFrameLevel() );
                    strong_highlight:GetPushedTexture():SetAllPoints( btn );
                    strong_highlight:SetButtonState( "pushed", true );
                    strong_highlight.row = self.row; -- to make deletable by ClearRows
                    strong_highlight.not_visible = true;
                    strong_highlight:Show();

                    count = count + 1;
                end
            end

            --[[ display notes ]]--
            SearchWindow:HideEditBox();
            tabulate_notes( result, DB, index, search_again );
        end,
        ["onenter"] = function( self )
            local count = 0;
            for btn in pairs( SearchWindow:GetRow( self.row ) ) do
                if not( btn.not_visible ) then
                    local highlight = SearchWindow.getframe["h"..count]["button:highlight"];
                    highlight:GetNormalTexture():SetAllPoints( btn );
                    highlight:SetFrameLevel( btn:GetFrameLevel() );
                    highlight:Show();
                    highlights[highlight] = true;
                    count = count + 1
                end
            end
        end,
        ["onleave"] = function()
            for highlight in pairs( highlights ) do
                highlight:Destroy();
            end

            highlights = {};
        end
    };

    SearchWindow:PrintRow( index, row );
end

local tabulate_data = function( results, DB, search_again )
    SearchWindow:ClearRows( 1, math.huge );

    num_results = TARGETNOTES_SETTINGS["number of result rows"];

    -- truncates length of results to num_results, as far as ipairs is concerned
    results[num_results+1] = nil;

    local last = 0;
    for index, result in ipairs( results ) do
        tabulate_result( index, result, DB, search_again );

        last = index;
    end

    while( last < num_results ) do
        last = last + 1;
        SearchWindow:PrintRow( last, blank_row );
    end

    -- prefill with blanks
    tabulate_notes();
    -- tabulate the notes of the top result
    tabulate_notes( (results or {})[1], DB, 1, search_again )
end

-- string prefixes
local prefixes = function( str )
    return coroutine.wrap( function()
        for i = 0, #str do
            coroutine.yield( string.sub( str, 1, i ) );
        end
    end );
end

tabulate_notes = function( result, DB, source_row, search_again )
    num_notes = TARGETNOTES_SETTINGS["number of note rows"];

    if( result ) then
        local server = result["entry"][1]["server"];
        if( (server ~= nil) and -- non-players units have server as nil
            (server ~= TARGETNOTES_FUNCTIONS["active_server"]) ) then
            server = " - " .. server;
        else
            server = "";
        end
        SearchWindow.notes_title_bar:SetText( result["entry"][1]["name"] .. server );
    else
        result = { ["entry"] = { 1, {} } };
    end

    SearchWindow:ClearRows( num_results + 1, math.huge );

    local last, prev = 0;
    for index, note in ipairs( result["entry"][2] ) do
        local row = get_note_row_template(
            { text = " " .. note,
              justification = "left",
              handlers = {
                    ["onclick"] = function( self )
                        SearchWindow.popup_menu:Disappear()

                        local rel_pos = GetCursorPosition() - self:GetLeft();
                        local position; -- figure out approximately which char user clicked on
                        for prefix in prefixes( "" .. note ) do
                            self:SetText( prefix );
                            if( self:GetTextWidth() > rel_pos ) then
                                position = #prefix - 1;
                                break;
                            end
                        end

                        SearchWindow:HideEditBox();
                        self:SetText( "" );

                        local editbox = SearchWindow:PopEditBox( self, {
                            ["onshow"] = function( self1 )
                                self1:SetFrameLevel( self:GetFrameLevel() + 1 );
                            end,
                            ["onhide"] = function()
                                self:SetText( " " .. (result["entry"][2][index] or "") );
                            end,
                            ["onescapepressed"] = function( self1 )
                                self:SetText( " " .. note );
                                SearchWindow:HideEditBox();
                            end,
                            ["onenterpressed"] = function( self1 )
                                if( self1:GetText() ~= note ) then -- note was edited
                                    local id, notes = result["id"], result["entry"][2];

                                    DB:RemoveEntry( id );
                                    table.remove( notes, index );
                                    table.insert( notes, index, self1:GetText() );

                                    DB:AddEntry( id, result["entry"] );

                                    TARGETNOTES_FUNCTIONS["update_target_notes"]();
                                    tabulate_result( source_row, result, DB, search_again );
                                    tabulate_notes( result, DB, source_row, search_again );
                                end

                                SearchWindow:HideEditBox();
                            end,
                            ["oneditfocuslost"] = function()
                                SearchWindow:HideEditBox();
                            end
                        } );

                        editbox:SetText( note )
                        editbox:SetCursorPosition( position or 999 );
                    end } },

            --[[ note delete button handlers ]]--
            { text = "",
              handlers = {
                    ["onclick"] = function( self )
                        local id, entry = result["id"], result["entry"];
                        DB:RemoveEntry( id );

                        table.remove( entry[2], index );
                        DB:AddEntry( id, entry );

                        SearchWindow:HideEditBox();

                        TARGETNOTES_FUNCTIONS["update_target_notes"]();
                        tabulate_result( source_row, result, DB, search_again );
                        tabulate_notes( result, DB, source_row, search_again );
                    end,
                    ["onenter"] = function( self )
                        self:SetText( "[delete]" )
                    end,
                    ["onleave"] = function( self )
                        self:SetText( "" )
                    end } }
        );

        --[[ whole row handlers ]]--
        row.handlers = {
                ["onenter"] = function( self )
                    --[[ highlighting ]]--
                    local count = 0;
                    for btn in pairs( SearchWindow:GetRow( self.row ) ) do
                        if not( btn.not_visible ) then
                            local highlight = SearchWindow.getframe["h"..count]["button:highlight"];
                            highlight:GetNormalTexture():SetAllPoints( btn );
                            highlight:SetFrameLevel( btn:GetFrameLevel() );
                            highlight:Show();
                            highlights[highlight] = true;
                            count = count + 1
                        end
                    end
                end,
                ["onleave"] = function()
                    for highlight in pairs( highlights ) do
                        highlight:Destroy();
                    end

                    highlights = {};
                end
        };

        prev = SearchWindow:PrintRow( num_results + 1 + index, row );

        last = index;
    end

    local add_blank_note_row = function( row )
        local row_data = get_note_row_template{ text = "", handlers = {
            ["onclick"] = function( self )
                SearchWindow.popup_menu:Disappear()
                SearchWindow:HideEditBox();
                SearchWindow:PopEditBox( self, {
                    ["onshow"] = function( self1 )
                        self1:SetFrameLevel( self:GetFrameLevel() + 1 );
                    end,
                    ["onenterpressed"] = function( self )
                        local id, entry = result["id"], result["entry"];
                        DB:RemoveEntry( id );

                        table.insert( entry[2], self:GetText() );
                        DB:AddEntry( id, entry );

                        SearchWindow:HideEditBox();

                        TARGETNOTES_FUNCTIONS["update_target_notes"]();
                        tabulate_result( source_row, result, DB, search_again );
                        tabulate_notes( result, DB, source_row, search_again );
                    end,
                    ["onescapepressed"] = function()
                        SearchWindow:HideEditBox();
                    end,
                    ["oneditfocuslost"] = function()
                        SearchWindow:HideEditBox();
                    end
                } );
            end
        } };

        return SearchWindow:PrintRow( row, row_data );
    end

    local freeze = last;
    SearchWindow.add_note_btn:SetScript( "onclick", function( self )
        SearchWindow.popup_menu:Disappear()
        local get_first_blank = function()
            for btn in pairs( SearchWindow:GetRow( num_results + 2 + freeze ) ) do
                if( btn:GetScript("onclick") ) then -- highlight and blank delete cell don't have onclick
                    btn:GetScript("onclick")( btn );
                    return true; -- return
                end
            end
        end

        if not( get_first_blank() ) then
            local prev = add_blank_note_row( num_results + 2 + freeze, row );
            SearchWindow.left_bar:SetPoint( "bottom", prev, "bottom", 0, 0 );
            get_first_blank();
        end
    end );


    while( last < num_notes ) do
        last = last + 1;

        prev = add_blank_note_row( num_results + 1 + last );
    end

    SearchWindow.left_bar:SetPoint( "bottom", prev, "bottom", 0, 0 );
end

TARGETNOTES_FUNCTIONS["tabulate_data"] = tabulate_data;
